package ga.core.individual;

import ga.core.algorithm.util.RandomSingleton;
import ga.core.validation.GAContext;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Random;

/**
 * Comparator for sorting and comparing individuals.
 * <p/>
 * 
 * Contains interval comparation method from <i>D. Gong, G. Guo,
 * "Interactive Genetic Algorithms with Interval Fitness of Evolutionary Individuals"
 * </i>.
 * 
 * @param <T>
 *          The generic type of individuals.
 * 
 * @since 31.01.2012
 * @author Stephan Dreyer
 */
class IndividualComparator<T extends IIndividual<T>> implements Comparator<T>,
    Serializable {
  private static final long serialVersionUID = 6409241438814158534L;
  private final boolean descending;

  private final Random rnd = RandomSingleton.getRandom();

  static {
    // this is required for JRE 7, otherwise the non-deterministic sort would
    // throw an exception
    System.setProperty("java.util.Arrays.useLegacyMergeSort", "true");
  }

  /**
   * Create a new comparator.
   * 
   * @param descending
   *          Sort descending, if <code>true</code>.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public IndividualComparator(final boolean descending) {
    this.descending = descending;
  }

  @Override
  public int compare(final T o1, final T o2) {
    if (!o1.isEvaluated()) {
      if (!o2.isEvaluated()) {
        return 0;
      } else {
        return 1;
      }
    }

    if (!o2.isEvaluated()) {
      if (o1.isEvaluated()) {
        return -1;
      }
    }

    int n;
    if (o1 instanceof IIntervalFitness && o2 instanceof IIntervalFitness) {
      n = compareInterval((IIntervalFitness) o1, (IIntervalFitness) o2);
    } else if (o1 instanceof IFitness && o2 instanceof IFitness) {
      n = compareNumber(o1, o2);
    } else {
      throw new RuntimeException("This individual has no fitness");
    }

    return descending ? -n : n;
  }

  /**
   * Simply compare point fitness.
   * 
   * @param o1
   *          Fitness 1
   * @param o2
   *          Fitness 2
   * @return a negative integer, zero, or a positive integer as the first
   *         argument is less than, equal to, or greater than the second.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  private int compareNumber(final IFitness o1, final IFitness o2) {
    final double d1 = o1.getFitness();
    final double d2 = o2.getFitness();

    return (d1 < d2) ? -1 : (d1 > d2) ? 1 : 0;
  }

  /**
   * Compare fitness intervals. This is a probabilistic method, so it creates no
   * stable order.
   * 
   * @param o1
   *          Fitness 1
   * @param o2
   *          Fitness 2
   * @return a negative integer, zero, or a positive integer as the first
   *         argument is less than, equal to, or greater than the second.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  private int compareInterval(final IIntervalFitness o1,
      final IIntervalFitness o2) {
    final double d = getDominationProbability(o1, o2);

    if (Math.abs(d - 0.5) < 0.000001d) {
      return 0;
    }

    return rnd.nextDouble() < d ? 1 : -1;
  }

  /**
   * Calculates the probability that one interval dominates another.
   * 
   * @param o1
   *          The first interval.
   * @param o2
   *          The second interval.
   * @return The domination probability.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  private double getDominationProbability(final IIntervalFitness o1,
      final IIntervalFitness o2) {
    final double d1 = o1.getFitness();
    final double d1Min = o1.getMinFitness();
    final double d1Max = o1.getMaxFitness();
    final double d1Width = o1.getFitnessWidth();

    final double d2 = o2.getFitness();
    final double d2Min = o2.getMinFitness();
    final double d2Max = o2.getMaxFitness();
    final double d2Width = o2.getFitnessWidth();

    if (d1Width == 0d && d2Width == 0d) {
      if (d1 == d2) {
        return 0.5d;
      }

      return d1 > d2 ? 1d : 0d;
    }

    // case 1
    if (d1Max > d2Max) {
      if (d1Min >= d2Max) {
        // I
        return 1d;
      } else {
        // II

        return 1d - (0.5d * ((d2Max - d1Min) / d2Width)//
        * ((d2Max - d1Min) / d1Width));
      }
    }

    // case 2
    if (d1Max <= d2Max && d1Min >= d2Min) {
      return (0.5d * d1Width / d2Width) + ((d1Min - d2Min) / d2Width);
    } else {
      return (0.5d * d2Width / d1Width) + ((d2Min - d1Min) / d1Width);
    }
  }

  /**
   * A main class to test behavior.
   * 
   * @param args
   *          Args
   * 
   * @since 10.08.2012
   * @author Stephan Dreyer
   */
  public static void main(final String[] args) {
    final IndividualComparator<TestInterval> cmp = new IndividualComparator<TestInterval>(
        true);
    final TestInterval x1 = new TestInterval("x1");
    final TestInterval x2 = new TestInterval("x2");
    final List<TestInterval> list = new ArrayList<TestInterval>();
    list.add(x1);
    list.add(x2);

    // TEST 1
    // ......|-----|
    // |-----|......
    // 0.1.2.3.4.5.6
    x1.setFitnessLimits(3d, 6d);
    x2.setFitnessLimits(0d, 3d);

    double p = cmp.getDominationProbability(x1, x2);
    System.err.println();
    System.err.println("Test 1: " + p);
    System.err.println("Sorting: " + cmp.compare(x1, x2));
    Collections.sort(list, cmp);
    System.err.println(list);

    // TEST 2
    // ..|-------|..
    // |-----|......
    // 0.1.2.3.4.5.6
    x1.setFitnessLimits(1d, 5d);
    x2.setFitnessLimits(0d, 3d);

    p = cmp.getDominationProbability(x1, x2);
    System.err.println();
    System.err.println("Test 2: " + p);
    System.err.println("Sorting: " + cmp.compare(x1, x2));
    Collections.sort(list, cmp);
    System.err.println(list);

    // TEST 3
    // |-----|......
    // |-----------|
    // 0.1.2.3.4.5.6
    x1.setFitnessLimits(0d, 3d);
    x2.setFitnessLimits(0d, 6d);

    p = cmp.getDominationProbability(x1, x2);
    System.err.println();
    System.err.println("Test 3: " + p);
    System.err.println("Sorting: " + cmp.compare(x1, x2));
    Collections.sort(list, cmp);
    System.err.println(list);

    // TEST 4
    // ......|-----|
    // |-----------|
    // 0.1.2.3.4.5.6
    x1.setFitnessLimits(3d, 6d);
    x2.setFitnessLimits(0d, 6d);

    p = cmp.getDominationProbability(x1, x2);
    System.err.println();
    System.err.println("Test 4: " + p);
    System.err.println("Sorting: " + cmp.compare(x1, x2));
    Collections.sort(list, cmp);
    System.err.println(list);

    // TEST 5
    // |-----------|
    // |-----|......
    // 0.1.2.3.4.5.6
    x1.setFitnessLimits(0d, 6d);
    x2.setFitnessLimits(0d, 3d);

    p = cmp.getDominationProbability(x1, x2);
    System.err.println();
    System.err.println("Test 5: " + p);
    System.err.println("Sorting: " + cmp.compare(x1, x2));
    Collections.sort(list, cmp);
    System.err.println(list);

    // TEST 6
    // |-----------|
    // ......|-----|
    // 0.1.2.3.4.5.6
    x1.setFitnessLimits(0d, 6d);
    x2.setFitnessLimits(3d, 6d);

    p = cmp.getDominationProbability(x1, x2);
    System.err.println();
    System.err.println("Test 6: " + p);
    System.err.println("Sorting: " + cmp.compare(x1, x2));
    Collections.sort(list, cmp);
    System.err.println(list);

    // TEST 7
    // ....|---|....
    // |-----------|
    // 0.1.2.3.4.5.6
    x1.setFitnessLimits(2d, 4d);
    x2.setFitnessLimits(0d, 6d);

    p = cmp.getDominationProbability(x1, x2);
    System.err.println();
    System.err.println("Test 7: " + p);
    System.err.println("Sorting: " + cmp.compare(x1, x2));
    Collections.sort(list, cmp);
    System.err.println(list);

    // TEST 8
    // |............
    // |-----------|
    // 0.1.2.3.4.5.6
    x1.setFitnessLimits(0d, 0d);
    x2.setFitnessLimits(0d, 6d);

    p = cmp.getDominationProbability(x1, x2);
    System.err.println();
    System.err.println("Test 8: " + p);
    System.err.println("Sorting: " + cmp.compare(x1, x2));
    Collections.sort(list, cmp);
    System.err.println(list);

    // TEST 9
    // ......|......
    // |-----------|
    // 0.1.2.3.4.5.6
    x1.setFitnessLimits(3d, 3d);
    x2.setFitnessLimits(0d, 6d);

    p = cmp.getDominationProbability(x1, x2);
    System.err.println();
    System.err.println("Test 9: " + p);
    System.err.println("Sorting: " + cmp.compare(x1, x2));
    Collections.sort(list, cmp);
    System.err.println(list);

    // TEST 10
    // ............|
    // |-----------|
    // 0.1.2.3.4.5.6
    x1.setFitnessLimits(10d, 10d);
    x2.setFitnessLimits(0d, 10d);

    p = cmp.getDominationProbability(x1, x2);
    System.err.println();
    System.err.println("Test 10: " + p);
    System.err.println("Sorting: " + cmp.compare(x1, x2));
    Collections.sort(list, cmp);
    System.err.println(list);

    // TEST 11
    // |............
    // ......|......
    // 0.1.2.3.4.5.6
    x1.setFitnessLimits(0d, 0d);
    x2.setFitnessLimits(3d, 3d);

    p = cmp.getDominationProbability(x1, x2);
    System.err.println();
    System.err.println("Test 11: " + p);
    System.err.println("Sorting: " + cmp.compare(x1, x2));
    Collections.sort(list, cmp);
    System.err.println(list);

    // TEST 12
    // ......|......
    // ......|......
    // 0.1.2.3.4.5.6
    x1.setFitnessLimits(3d, 3d);
    x2.setFitnessLimits(3d, 3d);

    p = cmp.getDominationProbability(x1, x2);
    System.err.println();
    System.err.println("Test 12: " + p);
    System.err.println("Sorting: " + cmp.compare(x1, x2));
    Collections.sort(list, cmp);
    System.err.println(list);

    // TEST 13
    // ............|
    // ......|......
    // 0.1.2.3.4.5.6
    x1.setFitnessLimits(6d, 6d);
    x2.setFitnessLimits(3d, 3d);

    p = cmp.getDominationProbability(x1, x2);
    System.err.println();
    System.err.println("Test 13: " + p);
    System.err.println("Sorting: " + cmp.compare(x1, x2));
    Collections.sort(list, cmp);
    System.err.println(list);
  }

  /**
   * This class is just for testing.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  private static class TestInterval implements IIntervalFitness,
      IIndividual<TestInterval> {
    private double min;
    private double max;
    private double center;
    private final String name;

    /**
     * 
     * @param name
     *          name
     * 
     * @since 11.08.2012
     * @author Stephan Dreyer
     */
    public TestInterval(final String name) {
      this.name = name;
    }

    @Override
    public void setFitness(final double fitness) {
      this.center = fitness;
    }

    @Override
    public double getFitness() {
      return center;
    }

    @Override
    public double getMinFitness() {
      return min;
    }

    @Override
    public double getMaxFitness() {
      return max;
    }

    @Override
    public double getFitnessWidth() {
      return max - min;
    }

    @Override
    public void setFitnessInterval(final double center, final double width) {
      this.center = center;
      min = center - (width / 2d);
      max = min + width;
    }

    @Override
    public void setFitnessLimits(final double min, final double max) {
      this.min = min;
      this.max = max;
      this.center = (max + min) / 2d;
    }

    @Override
    public void initRandomly() {
    }

    @Override
    public boolean isEvaluated() {
      return false;
    }

    @Override
    public long getId() {
      return 0L;
    }

    @Override
    public void setContext(final GAContext context) {
    }

    @Override
    public GAContext getContext() {
      return null;
    }

    @Override
    public TestInterval clone() {
      return new TestInterval(name);
    }

    @Override
    public String toString() {
      return name + " [" + min + ";" + max + "]";
    }

  }
}
